package com.bonc.xcloud.sp_procedure.util;

import org.apache.log4j.Logger;
import org.apache.log4j.PropertyConfigurator;

import com.bonc.xcloud.sp_procedure.backend.interpreter.executors.xcloud.ProcExceptionMessage;
import com.bonc.xcloud.sp_procedure.base.ProcParseException;
import com.bonc.xcloud.sp_procedure.frontend.Lexer;
import com.bonc.xcloud.sp_procedure.frontend.Source;
import com.bonc.xcloud.sp_procedure.frontend.StringSource;
import com.bonc.xcloud.sp_procedure.frontend.Token;
import com.bonc.xcloud.sp_procedure.frontend.xcloud.XcloudLexer;
import com.bonc.xcloud.sp_procedure.frontend.xcloud.XcloudParse;
import com.bonc.xcloud.sp_procedure.frontend.xcloud.XcloudTokenType;
import com.bonc.xcloud.sp_procedure.frontend.xcloud.plparsers.StatementParser;
import com.bonc.xcloud.sp_procedure.frontend.xcloud.sqlparser.InsertIntoFromClientParser;
import com.bonc.xcloud.sp_procedure.frontend.xcloud.sqlparser.InsertIntoStatmentParser;
import com.bonc.xcloud.sp_procedure.frontend.xcloud.sqlparser.UpsertStatmentParser;
import com.bonc.xcloud.sp_procedure.intermediate.syntax.sql.InsertIntoStatement;
import com.bonc.xcloud.sp_procedure.intermediate.syntax.sql.UpsertValueStatement;
import com.bonc.xcloud.sp_procedure.intermediate.syntax.syntaximpl.SyntaxTree;
import com.bonc.xcloud.sp_procedure.message.Message;
import com.bonc.xcloud.sp_procedure.message.MessageHandler;
import com.bonc.xcloud.sp_procedure.message.MessageListener;
import com.bonc.xcloud.sp_procedure.message.MessageType;

/**
 * 单条sql语句解析工具类
 */
public class SingleSqlParseUtil {
	
	private static Logger logger = Logger.getLogger(SourceMessageListener.class);
	
	//加载log配置文件
	static {
//		PropertyConfigurator.configure("conf/xcloudprocedure_log4j.properties");
	}
	
	/**
	 * 解析sql语句，返回语法树
	 */
	public static SyntaxTree parseSql(String sql) throws Exception{
		
		SyntaxTree syntaxTree = null;
		
		//无sql语句
		if(sql == null || "".equals(sql)) 
			throw toProcException("No sql statement can be found!");
		
		//源文件对象
		Source source = new StringSource(sql);
		source.setMessageHandler(new MessageHandler());
		source.addMessageListener(new SourceMessageListener());
		
		//词法解析器
		Lexer lexer = new XcloudLexer(source);
		//取出insert前的注释
		String annotition = lexer.skipAnnotition();
		//取出第一个单词
		Token token = lexer.nextToken();
		
		//第一个单词类型
		XcloudTokenType tokenType = (XcloudTokenType)token.getType();
		
		//语句解析器
		StatementParser statmentParser = null;
		
		//语法解析监听器
		ParserMessageListener parserListener = new ParserMessageListener();
		
		//判断使用对应的语句解析器
		switch (tokenType){
		
//			case SELECT:
//				statmentParser = new SelectParser(initParentParser(lexer,parserListener));
//				break;
			case INSERT:{
				statmentParser = new InsertIntoStatmentParser(initParentParser(lexer,parserListener));
				break;
			}
			//@Client ...
			case AT:{
				statmentParser = new InsertIntoFromClientParser(initParentParser(lexer,parserListener));
				break;
			}
			case UPSERT:{
				statmentParser = new UpsertStatmentParser(initParentParser(lexer,parserListener));
				break;
			}
			default:
				break;
		}
		
		if(statmentParser == null)
			throw toProcException("Unsupported Statement!!");
		
		try{
			//解析sql语句
			syntaxTree = statmentParser.parse(token);
			if(syntaxTree!=null){
				if(syntaxTree instanceof InsertIntoStatement){
					((InsertIntoStatement) syntaxTree).setAnnotation(annotition);
				}else if(syntaxTree instanceof UpsertValueStatement){
					((UpsertValueStatement) syntaxTree).setAnnotation(annotition);
				}
			}
		}catch(Exception e){
			//解析遇到意外的异常
			String errorMessage = e.getMessage()==null?e.toString():e.getMessage();
			throw toProcException(errorMessage);
		}
		
		//解析异常
		if(statmentParser.getErrorCount() != 0){
			//语法解析有错误
			String errorMessage = parserListener.getExceptionMessages();
			throw toProcException(errorMessage);
		}
		
		return syntaxTree;
	}
	
	/**
	 * 构造一个解析异常
	 */
	private static ProcParseException toProcException(String message){
		
		ProcParseException parseException = new ProcParseException();
		
		parseException.setDetailMessage("description: "+message);
		
		return parseException;
	}
	
	/**
	 * 初始化父解析器，父解析器负责初始化符号表，异常处理，消息监听器等
	 */
	private static XcloudParse initParentParser(Lexer lexer,ParserMessageListener parserListener){
		
		XcloudParse parser = new XcloudParse(lexer);
		//parser初始化
		parser.initialize();
		//添加parser监听器
		parser.addMessageListener(parserListener);
		
		return parser;
	}
	
	
	
	public static void main(String[] args) {
		
	}
	
	

}

/**
* 源文件消息监听器
*
*/
class SourceMessageListener implements MessageListener{
	
	private static Logger logger = Logger.getLogger(SourceMessageListener.class);
	private static final String SOURCE_LINE_FORMAT = "%03d %s";//源文件行格式
	@Override
	public void messageReceived(Message message) {
		
		MessageType type = message.getType();
		Object body[] = (Object[]) message.getBody();
		
		switch(type){
		
			case SOURCE_LINE:{
				int lineNumber = (Integer) body[0];
				String lineText = (String) body[1];
				
				logger.debug(String.format(SOURCE_LINE_FORMAT,
                      lineNumber, lineText));
			}
		}
	}
}

/**
* 语法解析器消息监听器
*
*/
class ParserMessageListener implements MessageListener{
	private static Logger logger = Logger.getLogger(SourceMessageListener.class);
	 private String exceptionMessages = "";
	private static final String PARSER_SUMMARY_FORMAT = //语法解析器总结格式
	        "\n%,20d source lines." +
	        "\n%,20d syntax errors." +
	        "\n%,20.2f seconds total parsing time.\n";
	
	private static final String TOKEN_FORMAT = // token的输出格式
			">>> %-15s line=%03d, pos=%2d, text=\"%s\"";
	
	private static final String VALUE_FORMAT = // token值的输出格式
			">>>                 value=%s";

	private static final int PREFIX_WIDTH = 5;
	
	@Override
	public void messageReceived(Message message) {
		
		MessageType type = message.getType();
		
		switch(type){
			
			case PARSER_SUMMARY:{
				Number body[] = (Number[]) message.getBody();
				int statementCount = (Integer) body[0]; // 语句个数
				int syntaxErrors = (Integer) body[1]; // 语法错误
				float elapsedTime = (Float) body[2]; // 时长
				
				logger.debug(String.format(PARSER_SUMMARY_FORMAT,
						statementCount,syntaxErrors,elapsedTime));
				 break; 
			}
			
//			case TOKEN:{
//				Object body[] = (Object []) message.getBody();
//				int line = (Integer) body[0];
//              int position = (Integer) body[1];
//              TokenType tokenType = (TokenType) body[2];
//              String tokenText = (String) body[3];
//              Object tokenValue = body[4];
//              logger.debug(String.format(TOKEN_FORMAT,
//                      tokenType,
//                      line,
//                      position,
//                      tokenText));
//              if(tokenValue != null){
//              	if (tokenType == XcloudTokenType.STRING) {
//                      tokenValue = "\"" + tokenValue + "\"";
//                  }
//
//                  logger.debug(String.format(VALUE_FORMAT,
//                                                   tokenValue));
//              }
//              
//              break;
//			}
			
			case SYNTAX_ERROR: {
				Object body[] = (Object []) message.getBody();
				int lineNumber = (Integer) body[0];
				int position = (Integer) body[1];
				String tokenText = (String) body[2];
				String errorMessage = (String) body[3];
				String procedureName = (String) body[4];
				
				int spaceCount = PREFIX_WIDTH + position;
				StringBuilder flagBuffer = new StringBuilder();
				
				// Spaces up to the error position.
              for (int i = 1; i < spaceCount; ++i) {
                  flagBuffer.append(' ');
              }
              
              // A pointer to the error followed by the error message.
              flagBuffer.append("^\n*** ").append(errorMessage);
				
              // Text, if any, of the bad token.
              if (tokenText != null) {
                  flagBuffer.append(" [at \"").append(tokenText)
                      .append("\"]");
              }
              
              logger.debug(flagBuffer.toString());
              exceptionMessages+=errorMessage+" at "+"Line:"+lineNumber+" Column:"+position+"\r\n";
              break;
				
			}
			
		}
	}

	
	public String getExceptionMessages() {
		return exceptionMessages;
	}

	public void setExceptionMessages(String exceptionMessages) {
		this.exceptionMessages = exceptionMessages;
	}
}











