package com.bonc.xcloud.sp_procedure.frontend;

import com.bonc.xcloud.sp_procedure.message.MessageHandler;

/**
 * 语言无关的Lexer，产生Token  
 */
public abstract class Lexer {
	
	protected Source source;// 源文件
	
	private Token currentToken;// 当前Token
	
	public Lexer(Source source){
		
		this.source = source;
	}
	/**
	 * 为了只处理字符串的这种情况
	 * @throws Exception 
	 */
//	public Lexer(StringSource stringSource){
//		this.source=stringSource;
//	}
	
	public Lexer(String sourceText) throws Exception{
		this.source = Source.newInstance
				(Source.stringSource,sourceText, new MessageHandler());
	}
	
	
	/**
	 * @return 当前Token
	 */
	public Token currentToken(){
		
		return currentToken;
	}
	
	/**
	 * @return 继续扫描,得到下一个Token
	 */
	public Token nextToken() throws Exception{
		
		currentToken = extractToken();
		
		return currentToken;
		
	}
	
	/**
	 * 因为每个源语言的Token构成方式不一样，所以这个具体语言的子类去实现。
	 * @return 语言相关的Token
	 * @throws Exception
	 */
	public abstract Token extractToken() throws Exception;
	
	/**
	 * @return 当前字符
	 * @throws Exception 
	 */
	public char currentChar() throws Exception{
		return source.currentChar();
	}
	
	/**
	 * @return 继续扫描,得到下一个字符
	 * @throws Exception 
	 */
	public char nextChar() throws Exception{
		return source.nextChar();
	}
	
	
	public Token peekToken() throws Exception{
		
		//缓存当前位置
		int current = source.getCurrentPos();
		int currentLine = source.getLineNum();
		String line = source.getLine();
		if(source instanceof FileSource)
			((FileSource)source).getReader().mark(100000);
		//读取下一个token
		Token token = extractToken();
		//设置位置为之前缓存的位置
		source.setCurrentPos(current);
		source.setLineNum(currentLine);
		source.setLine(line);
		if(source instanceof FileSource)
			((FileSource)source).getReader().reset();
		return token;
	}

	public Token peekToken(int index) throws Exception{
		
		//缓存当前位置
		int current = source.getCurrentPos();
		int currentLine = source.getLineNum();
		String line = source.getLine();
		if(source instanceof FileSource)
			((FileSource)source).getReader().mark(100000);
		//读取下一个token
		Token token = null;
		for(int i =0;i<index;i++){
			token = extractToken();
		}
		//设置位置为之前缓存的位置
		source.setCurrentPos(current);
		source.setLineNum(currentLine);
		source.setLine(line);
		if(source instanceof FileSource)
			((FileSource)source).getReader().reset();
		return token;
	}

	/**
	 * 跳过空白字符,注释部分
	 */
	public String skipAnnotition() throws Exception {
		
		StringBuffer buffer=new StringBuffer();
		char currentChar = currentChar();
		
		while(Character.isWhitespace(currentChar) || currentChar=='/' || currentChar == '-'){
			
			//可能是单行注释
			if(currentChar == '-'){
				char nextFirstChar = source.peekChar(1);
				//单行注释
				if(nextFirstChar == '-'){
					buffer.append("--");
					while(true){
						currentChar = nextChar();
						if(currentChar == Source.EOL){
							buffer.append(currentChar);
							currentChar = nextChar();
							break;
						}
					}
				}else{
					break;
				}
			}
			//可能是多行注释
			else if(currentChar=='/'){
				
				char nextFirstChar = source.peekChar(1);
				
				if(nextFirstChar == '*'){
					buffer.append("/*");
					char nextSecondChar = source.peekChar(2);
					
					//排除hint的可能,为多行注释
					if(nextSecondChar != '+'){
						//跳转到*后面的字母
						nextChar();
						while(true){
							currentChar = nextChar();
							buffer.append(currentChar);
							if(currentChar == Source.EOF || (currentChar == '*'&&(source.peekChar(1)=='/'))){
								buffer.append("/");
								currentChar = nextChar();
								currentChar = nextChar();
								break;
							}
						}
					}else{
						break;
					}
				}else{
					break;
				}
			}else{
				currentChar = nextChar();
			}
		}
		return buffer.toString();
			
	}
	
	public Source getSource() {
		return source;
	}

	public void setSource(Source source) {
		this.source = source;
	}
	

}
